<?php
/**
 * Post meta box functionality
 * 
 * @package DRC
 * @since 2.2.0
 */

// Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

class DRC_Post_Meta {
    /**
     * Initialize the class
     */
    public static function init() {
        add_action('add_meta_boxes', [__CLASS__, 'add_meta_box']);
        add_action('save_post', [__CLASS__, 'save_meta_box']);
    }

    /**
     * Add meta box to post editor
     */
    public static function add_meta_box() {
        $screens = ['post', 'page'];
        
        // Add support for custom post types
        $custom_types = get_post_types(['public' => true, '_builtin' => false]);
        $screens = array_merge($screens, $custom_types);

        foreach ($screens as $screen) {
            add_meta_box(
                'drc_post_settings',
                __('Right Click & Text Selection Settings', 'disable-right-click-and-select'),
                [__CLASS__, 'render_meta_box'],
                $screen,
                'side',
                'default'
            );
        }
    }

    /**
     * Render meta box content
     */
    public static function render_meta_box($post) {
        // Add nonce for security
        wp_nonce_field('drc_post_meta_box', 'drc_post_meta_box_nonce');

    $settings = drc_get_settings();
    $flags = drc_get_post_flags($post->ID);
    $disable_right_click = drc_flag_enabled($flags, DRC_FLAG_RIGHT_CLICK);
    $disable_text_selection = drc_flag_enabled($flags, DRC_FLAG_TEXT_SELECTION);
    $message = drc_get_post_message($post->ID, $settings);

        // Output the form fields
        ?>
        <div class="drc-meta-box-content">
            <p>
                <label>
              <input type="checkbox" 
                  name="drc_post_settings[disable_right_click]" 
                  value="1" 
                  <?php checked($disable_right_click); ?>>
                    <?php esc_html_e('Disable Right Click', 'disable-right-click-and-select'); ?>
                </label>
            </p>
            
            <p>
                <label>
              <input type="checkbox" 
                  name="drc_post_settings[disable_text_selection]" 
                  value="1" 
                  <?php checked($disable_text_selection); ?>>
                    <?php esc_html_e('Disable Text Selection', 'disable-right-click-and-select'); ?>
                </label>
            </p>
            
            <p>
                <label for="drc_right_click_message">
                    <?php esc_html_e('Right Click Message:', 'disable-right-click-and-select'); ?>
                </label>
                <input type="text" 
                       id="drc_right_click_message" 
                       name="drc_post_settings[right_click_message]" 
                       value="<?php echo esc_attr($message); ?>" 
                       class="widefat">
            </p>

            <p>
                <button type="button" 
                        class="button drc-reset-settings" 
                        data-post-id="<?php echo esc_attr($post->ID); ?>">
                    <?php esc_html_e('Reset to Default Settings', 'disable-right-click-and-select'); ?>
                </button>
            </p>
        </div>
        <?php
    }

    /**
     * Save meta box data
     */
    public static function save_meta_box($post_id) {
        // Security checks
        if (!isset($_POST['drc_post_meta_box_nonce'])) {
            return;
        }

        $nonce = sanitize_text_field(wp_unslash($_POST['drc_post_meta_box_nonce']));
        if (!wp_verify_nonce($nonce, 'drc_post_meta_box')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        $settings = drc_get_settings();
        // Retrieve posted settings safely without direct superglobal access
        $posted = filter_input_array(INPUT_POST, [
            'drc_post_settings' => [ 'filter' => FILTER_DEFAULT, 'flags' => FILTER_REQUIRE_ARRAY ],
        ]);
        if (isset($posted['drc_post_settings']) && is_array($posted['drc_post_settings'])) {
            $raw = wp_unslash($posted['drc_post_settings']);
            // Sanitize array values defensively
            $disable_right_click = !empty($raw['disable_right_click']) ? 1 : 0;
            $disable_text_selection = !empty($raw['disable_text_selection']) ? 1 : 0;
            $message = isset($raw['right_click_message']) ? sanitize_text_field($raw['right_click_message']) : '';
            $flags = drc_encode_flags($disable_right_click, $disable_text_selection);
            drc_update_post_flags($post_id, $flags, $message, $settings['right_click_message']);
        } else {
            // No settings posted: remove all
            drc_update_post_flags($post_id, 0, '');
        }
    }
}

// Initialize the class
DRC_Post_Meta::init(); 