<?php
/**
 * Post list functionality
 * 
 * @package DRC
 * @since 2.2.0
 */

// Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

class DRC_Post_List {
    /**
     * Initialize the class
     */
    public static function init() {
        // Add columns to post list
        add_filter('manage_posts_columns', [__CLASS__, 'add_posts_column']);
        add_filter('manage_pages_columns', [__CLASS__, 'add_posts_column']);
        
        // Add column content
        add_action('manage_posts_custom_column', [__CLASS__, 'render_column'], 10, 2);
        add_action('manage_pages_custom_column', [__CLASS__, 'render_column'], 10, 2);
        
        // Add quick edit support
        add_action('quick_edit_custom_box', [__CLASS__, 'add_quick_edit'], 10, 2);
        
        // Add bulk edit support
        add_action('bulk_edit_custom_box', [__CLASS__, 'add_bulk_edit'], 10, 2);
        
        // Save quick edit data
        add_action('save_post', [__CLASS__, 'save_quick_edit_data']);
        
        // Save bulk edit data
        add_action('wp_ajax_drc_save_bulk_edit', [__CLASS__, 'save_bulk_edit_data']);
        
        // Add column styles
        add_action('admin_head', [__CLASS__, 'add_column_styles']);
    }

    /**
     * Add custom column to posts list
     */
    public static function add_posts_column($columns) {
        $columns['drc_protection'] = __('Protection Status', 'disable-right-click-and-select');
        return $columns;
    }

    /**
     * Render column content
     */
    public static function render_column($column_name, $post_id) {
        if ($column_name !== 'drc_protection') {
            return;
        }

    $flags = drc_get_post_flags($post_id);
    if (!$flags) {
            printf(
                '<span class="drc-status-none">%s</span>',
                esc_html__('Not Protected', 'disable-right-click-and-select')
            );
            return;
        }

        $status_items = [];
    if (drc_flag_enabled($flags, DRC_FLAG_RIGHT_CLICK)) {
            $status_items[] = sprintf(
                '<span class="drc-status-icon dashicons dashicons-lock"></span>%s',
                esc_html__('Right Click Disabled', 'disable-right-click-and-select')
            );
        }
    if (drc_flag_enabled($flags, DRC_FLAG_TEXT_SELECTION)) {
            $status_items[] = sprintf(
                '<span class="drc-status-icon dashicons dashicons-shield"></span>%s',
                esc_html__('Text Selection Disabled', 'disable-right-click-and-select')
            );
        }

        echo wp_kses(
            implode('<br>', $status_items),
            array(
                'span' => array(
                    'class' => array()
                ),
                'br' => array()
            )
        );
    }

    /**
     * Add quick edit fields
     */
    public static function add_quick_edit($column_name, $post_type) {
        if ($column_name !== 'drc_protection') {
            return;
        }
        // Add nonce field
        wp_nonce_field('drc_quick_edit_nonce', 'drc_quick_edit_nonce');
        ?>
        <fieldset class="inline-edit-col-right drc-quick-edit">
            <div class="inline-edit-col">
                <label class="inline-edit-group">
                    <span class="title"><?php esc_html_e('Protection Settings', 'disable-right-click-and-select'); ?></span>
                    <label>
                        <input type="checkbox" name="drc_disable_right_click" value="1">
                        <span class="checkbox-title"><?php esc_html_e('Disable Right Click', 'disable-right-click-and-select'); ?></span>
                    </label>
                    <label>
                        <input type="checkbox" name="drc_disable_text_selection" value="1">
                        <span class="checkbox-title"><?php esc_html_e('Disable Text Selection', 'disable-right-click-and-select'); ?></span>
                    </label>
                </label>
            </div>
        </fieldset>
        <?php
    }

    /**
     * Add bulk edit fields
     */
    public static function add_bulk_edit($column_name, $post_type) {
        if ($column_name !== 'drc_protection') {
            return;
        }
        ?>
        <fieldset class="inline-edit-col-right drc-bulk-edit">
            <div class="inline-edit-col">
                <label class="inline-edit-group">
                    <span class="title"><?php esc_html_e('Protection Settings', 'disable-right-click-and-select'); ?></span>
                    <label>
                        <select name="drc_disable_right_click">
                            <option value="-1"><?php esc_html_e('— No Change —', 'disable-right-click-and-select'); ?></option>
                            <option value="1"><?php esc_html_e('Enable', 'disable-right-click-and-select'); ?></option>
                            <option value="0"><?php esc_html_e('Disable', 'disable-right-click-and-select'); ?></option>
                        </select>
                        <span class="checkbox-title"><?php esc_html_e('Right Click Protection', 'disable-right-click-and-select'); ?></span>
                    </label>
                    <label>
                        <select name="drc_disable_text_selection">
                            <option value="-1"><?php esc_html_e('— No Change —', 'disable-right-click-and-select'); ?></option>
                            <option value="1"><?php esc_html_e('Enable', 'disable-right-click-and-select'); ?></option>
                            <option value="0"><?php esc_html_e('Disable', 'disable-right-click-and-select'); ?></option>
                        </select>
                        <span class="checkbox-title"><?php esc_html_e('Text Selection Protection', 'disable-right-click-and-select'); ?></span>
                    </label>
                </label>
            </div>
        </fieldset>
        <?php
    }

    /**
     * Save quick edit data
     */
    public static function save_quick_edit_data($post_id) {
        // Security checks
        if (!isset($_POST['drc_quick_edit_nonce'])) {
            return;
        }

        $nonce = sanitize_text_field(wp_unslash($_POST['drc_quick_edit_nonce']));
        if (!wp_verify_nonce($nonce, 'drc_quick_edit_nonce')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Validate and sanitize input
        $disable_right_click = isset($_POST['drc_disable_right_click']) ? 
            sanitize_text_field(wp_unslash($_POST['drc_disable_right_click'])) : '';
        $disable_text_selection = isset($_POST['drc_disable_text_selection']) ? 
            sanitize_text_field(wp_unslash($_POST['drc_disable_text_selection'])) : '';

        if (empty($disable_right_click) && empty($disable_text_selection)) {
            return;
        }

    $disable_rc = ($disable_right_click === '1');
    $disable_ts = ($disable_text_selection === '1');
    $flags = drc_encode_flags($disable_rc, $disable_ts);
    drc_update_post_flags($post_id, $flags);        
    }

    /**
     * Save bulk edit data
     */
    public static function save_bulk_edit_data() {
        check_ajax_referer('drc-bulk-edit', 'security');
        if (!current_user_can('edit_posts')) { wp_send_json_error(['message'=>'forbidden'],403); }
        if (!isset($_POST['post_ids']) || !is_array($_POST['post_ids'])) { wp_send_json_error(['message'=>'invalid_post_ids'],400); }
        $post_ids = array_map('absint', wp_unslash($_POST['post_ids']));
        $right_click = isset($_POST['right_click']) ? absint(wp_unslash($_POST['right_click'])) : -1;
        $text_selection = isset($_POST['text_selection']) ? absint(wp_unslash($_POST['text_selection'])) : -1;
        if (!in_array($right_click, [-1,0,1], true) || !in_array($text_selection, [-1,0,1], true)) { wp_send_json_error(['message'=>'invalid_flags'],400); }
        $updated = [];
        foreach ($post_ids as $post_id) {
            if (!$post_id || !current_user_can('edit_post',$post_id) || !get_post($post_id)) { continue; }
            $existing = drc_get_post_flags($post_id);
            $disable_rc = drc_flag_enabled($existing, DRC_FLAG_RIGHT_CLICK);
            $disable_ts = drc_flag_enabled($existing, DRC_FLAG_TEXT_SELECTION);
            if ($right_click !== -1) { $disable_rc = (bool)$right_click; }
            if ($text_selection !== -1) { $disable_ts = (bool)$text_selection; }
            $new_flags = drc_encode_flags($disable_rc,$disable_ts);
            if ($new_flags !== $existing) { drc_update_post_flags($post_id,$new_flags); $updated[]=$post_id; }
        }
        wp_send_json_success(['updated'=>$updated,'count'=>count($updated)]);
    }

    /**
     * Add column styles
     */
    public static function add_column_styles() {
        ?>
        <style type="text/css">
            .column-drc_protection {
                width: 15%;
            }
            .drc-status-icon {
                color: #2271b1;
                margin-right: 5px;
            }
            .drc-status-none {
                color: #666;
                font-style: italic;
            }
            .drc-quick-edit .inline-edit-group span.title {
                width: 100%;
                float: none;
                margin: 0 0 10px;
            }
            .drc-quick-edit .inline-edit-group label {
                display: block;
                margin: 5px 0;
            }
            .drc-bulk-edit .inline-edit-group span.title {
                width: 100%;
                float: none;
                margin: 0 0 10px;
            }
            .drc-bulk-edit .inline-edit-group label {
                display: block;
                margin: 5px 0;
            }
            .drc-bulk-edit .inline-edit-group select {
                margin-right: 10px;
            }
        </style>
        <?php
    }
}

// Initialize the class
DRC_Post_List::init(); 