<?php
/**
 * Core bootstrap: constants, activation/deactivation, i18n, helpers
 */

if (!defined('ABSPATH')) { exit; }

// Constants are already defined in main file (back-compat). Provide guard.
if (!defined('DRC_VERSION')) {
    define('DRC_VERSION', '2.3.1');
}

function drc_default_settings() {
    return [
        'disable_right_click'    => false,
        'disable_text_selection' => false,
        'right_click_message'    => __('This action is not allowed.', 'disable-right-click-and-select'),
        'disable_on_post_types'  => ['post','page'],
        'disable_for_roles'      => ['guest'],
        'disabled_posts'         => [] // legacy
    ];
}

function drc_get_settings() {
    $defaults = drc_default_settings();
    $saved = get_option('drc_settings', []);
    $settings = wp_parse_args($saved, $defaults);
    $settings['disable_right_click']    = (bool)$settings['disable_right_click'];
    $settings['disable_text_selection'] = (bool)$settings['disable_text_selection'];
    $settings['disable_on_post_types']  = (array)$settings['disable_on_post_types'];
    $settings['disable_for_roles']      = (array)$settings['disable_for_roles'];
    return $settings;
}

function drc_activate($network_wide = false) {
    if (false === get_option('drc_settings')) {
        add_option('drc_settings', drc_default_settings());
    }
}
register_activation_hook(DRC_PLUGIN_FILE, 'drc_activate');

function drc_deactivate($network_wide = false) {
    delete_transient('drc_cache');
}
register_deactivation_hook(DRC_PLUGIN_FILE, 'drc_deactivate');

function drc_uninstall() {
    delete_option('drc_settings');
    delete_transient('drc_plugin_settings');
    if (function_exists('wp_cache_flush')) { wp_cache_flush(); }
}
register_uninstall_hook(DRC_PLUGIN_FILE, 'drc_uninstall');

// Bitmask constants
if (!defined('DRC_FLAG_RIGHT_CLICK')) { define('DRC_FLAG_RIGHT_CLICK', 1); }
if (!defined('DRC_FLAG_TEXT_SELECTION')) { define('DRC_FLAG_TEXT_SELECTION', 2); }

function drc_encode_flags($rc, $ts) { $f=0; if($rc){$f|=DRC_FLAG_RIGHT_CLICK;} if($ts){$f|=DRC_FLAG_TEXT_SELECTION;} return $f; }
function drc_flag_enabled($flags,$flag){ return ((int)$flags & (int)$flag) === (int)$flag; }

function drc_get_post_flags($post_id) {
    $post_id = (int)$post_id; if(!$post_id){ return 0; }
    $flags = get_post_meta($post_id, '_drc_flags', true);
    if ($flags !== '' && $flags !== false) { return (int)$flags; }
    $settings = drc_get_settings();
    if (isset($settings['disabled_posts'][$post_id])) {
        $legacy = $settings['disabled_posts'][$post_id];
        return drc_encode_flags(!empty($legacy['disable_right_click']), !empty($legacy['disable_text_selection']));
    }
    return 0;
}

function drc_get_post_message($post_id, $settings = null) {
    $post_id = (int)$post_id; if($settings===null){ $settings = drc_get_settings(); }
    if ($post_id) {
        $msg = get_post_meta($post_id, '_drc_message', true);
        if (is_string($msg) && $msg !== '') { return $msg; }
        if (isset($settings['disabled_posts'][$post_id]['right_click_message'])) { return $settings['disabled_posts'][$post_id]['right_click_message']; }
    }
    return $settings['right_click_message'];
}

function drc_update_post_flags($post_id, $flags, $message = null, $global_message = null) {
    $post_id = (int)$post_id; if(!$post_id){return;}
    $flags = (int)$flags;
    if ($flags === 0) { delete_post_meta($post_id, '_drc_flags'); } else { update_post_meta($post_id, '_drc_flags', $flags); }
    if ($message !== null) {
        if ($global_message === null) { $settings = drc_get_settings(); $global_message = $settings['right_click_message']; }
        if ($message === '' || $message === $global_message) { delete_post_meta($post_id, '_drc_message'); }
        else { update_post_meta($post_id, '_drc_message', $message); }
    }
}

function drc_maybe_migrate_disabled_posts() {
    $settings = drc_get_settings();
    if (empty($settings['disabled_posts'])) { return; }
    $global_message = $settings['right_click_message'];
    foreach ($settings['disabled_posts'] as $post_id => $legacy) {
        $post_id = (int)$post_id; if(!$post_id || !get_post($post_id)){ continue; }
        $flags = drc_encode_flags(!empty($legacy['disable_right_click']), !empty($legacy['disable_text_selection']));
        $msg = isset($legacy['right_click_message']) ? $legacy['right_click_message'] : $global_message;
        drc_update_post_flags($post_id, $flags, $msg, $global_message);
    }
    $settings['disabled_posts'] = [];
    update_option('drc_settings', $settings, false);
}
add_action('plugins_loaded', 'drc_maybe_migrate_disabled_posts', 20);
